<?php

/**
* @package     Tag Groups
* @author      Christoph Amthor
* @copyright   2018 Christoph Amthor (@ Chatty Mango, chattymango.com)
* @license     GPL-3.0+
*/
if ( !class_exists( 'TagGroups_Settings' ) ) {
    /**
     *
     */
    class TagGroups_Settings
    {
        function __construct()
        {
        }
        
        /**
         * renders the top of all setting pages
         *
         * @param void
         * @return void
         */
        public static function add_header()
        {
            $view = new TagGroups_View( 'admin/settings_header' );
            $view->set( 'admin_page_title', get_admin_page_title() );
            $view->render();
        }
        
        /**
         * renders the bottom of all settings pages
         *
         * @param void
         * @return void
         */
        public static function add_footer()
        {
            $view = new TagGroups_View( 'admin/settings_footer' );
            $view->render();
        }
        
        /**
         * gets the slug of the currently selected tab
         *
         * @param string $default
         * @return string
         */
        public static function get_active_tab( $tabs )
        {
            
            if ( isset( $_GET['active-tab'] ) ) {
                return sanitize_title( $_GET['active-tab'] );
            } else {
                $keys = array_keys( $tabs );
                return reset( $keys );
            }
        
        }
        
        /**
         * gets the HTML of the header of tabbed view
         *
         * @param string $default
         * @return string
         */
        public static function add_tabs( $page, $tabs, $active_tab )
        {
            if ( count( $tabs ) < 2 ) {
                return ( empty($label) ? '' : '<h2>' . $label . '</h2>' );
            }
            $view = new TagGroups_View( 'admin/settings_tabs' );
            $view->set( array(
                'tabs'       => $tabs,
                'page'       => $page,
                'active_tab' => $active_tab,
            ) );
            $view->render();
        }
        
        /**
         * renders a settings page: home
         *
         * @param void
         * @return void
         */
        public static function settings_page_home()
        {
            global  $tag_group_groups ;
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
            self::add_header();
            $html = '';
            $tg_group = new TagGroups_Group();
            $group_count = $tag_group_groups->get_number_of_term_groups();
            $tag_group_base_first_activation_time = TagGroups_Options::get_option( 'tag_group_base_first_activation_time', 0 );
            $tag_group_premium_first_activation_time = TagGroups_Options::get_option( 'tag_group_base_first_activation_time', 0 );
            $absolute_first_activation_time = ( $tag_group_base_first_activation_time < $tag_group_premium_first_activation_time ? $tag_group_base_first_activation_time : $tag_group_premium_first_activation_time );
            self::add_settings_help();
            $alerts = array();
            
            if ( 'all' == TagGroups_WPML::get_current_language() ) {
                $view = new TagGroups_View( 'partials/language_notice' );
                $view->render();
            }
            
            if ( time() - $absolute_first_activation_time < 60 * 60 * 24 * 7 || $group_count < 2 ) {
                $alerts[] = sprintf( __( 'See the <a %s>First Steps</a> for some basic instructions on how to get started.', 'tag-groups' ), 'href="' . menu_page_url( 'tag-groups-settings-first-steps', false ) . '"' );
            }
            
            if ( function_exists( 'pll_get_post_language' ) ) {
                $alerts[] = __( 'We detected Polylang. Your tag group names are translatable.', 'tag-groups' );
            } elseif ( defined( 'ICL_LANGUAGE_CODE' ) ) {
                $alerts[] = __( 'We detected WPML. Your tag group names are translatable.', 'tag-groups' );
            }
            
            $alerts = apply_filters( 'tag_groups_settings_alerts', $alerts );
            
            if ( !empty($alerts) ) {
                $view = new TagGroups_View( 'partials/settings_alerts' );
                $view->set( 'alerts', $alerts );
                $view->render();
            }
            
            $taxonomy_infos = array();
            foreach ( $enabled_taxonomies as $taxonomy ) {
                /**
                 * We try to avoid excessive loading times on this page
                 */
                $term_count = get_terms( array(
                    'hide_empty' => false,
                    'taxonomy'   => $taxonomy,
                    'fields'     => 'count',
                ) );
                if ( is_object( $term_count ) ) {
                    continue;
                }
                $tag_groups_info_obj = new TagGroups_Shortcode_Info();
                $taxonomy_infos[] = array(
                    'slug'            => $taxonomy,
                    'tag_group_admin' => TagGroups_Taxonomy::get_tag_group_admin_url( $taxonomy ),
                    'name'            => TagGroups_Taxonomy::get_name_from_slug( $taxonomy ),
                    'info_html'       => $tag_groups_info_obj->tag_groups_info( array(
                    'taxonomy'   => $taxonomy,
                    'group_id'   => 'all',
                    'html_class' => 'widefat fixed striped',
                ) ),
                    'term_count'      => $term_count,
                );
            }
            $view = new TagGroups_View( 'admin/settings_home' );
            $view->set( array(
                'group_count'    => $group_count,
                'taxonomy_infos' => $taxonomy_infos,
            ) );
            $view->render();
            self::add_footer();
        }
        
        /**
         * renders a settings page: taxonomies
         *
         * @param void
         * @return void
         */
        public static function settings_page_taxonomies()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
            $public_taxonomies = TagGroups_Taxonomy::get_public_taxonomies();
            self::add_header();
            $html = '';
            self::add_settings_help();
            $tabs = array();
            $tabs['taxonomies'] = '';
            $tabs = apply_filters( 'tag_groups_settings_taxonomies_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-taxonomies', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'taxonomies':
                    $view = new TagGroups_View( 'admin/settings_taxonomies' );
                    $view->set( array(
                        'public_taxonomies'  => $public_taxonomies,
                        'enabled_taxonomies' => $enabled_taxonomies,
                    ) );
                    $view->render();
                    break;
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * renders a settings page: back end
         *
         * @param void
         * @return void
         */
        public static function settings_page_back_end()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            self::add_header();
            self::add_settings_help();
            $tabs = array();
            $tabs['filters'] = __( 'Filters', 'tag-groups' );
            if ( TagGroups_Gutenberg::is_gutenberg_active() ) {
                $tabs['gutenberg'] = __( 'Gutenberg', 'tag-groups' );
            }
            $tabs = apply_filters( 'tag_groups_settings_back_end_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-back-end', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'filters':
                    $show_filter_posts = TagGroups_Options::get_option( 'tag_group_show_filter', 1 );
                    $show_filter_tags = TagGroups_Options::get_option( 'tag_group_show_filter_tags', 1 );
                    $view = new TagGroups_View( 'admin/settings_back_end_filters' );
                    $view->set( array(
                        'show_filter_posts' => $show_filter_posts,
                        'show_filter_tags'  => $show_filter_tags,
                    ) );
                    $view->render();
                    break;
                    // filters
                // filters
                case 'gutenberg':
                    $tag_group_server_side_render = TagGroups_Options::get_option( 'tag_group_server_side_render', 1 );
                    $view = new TagGroups_View( 'admin/settings_back_end_gutenberg' );
                    $view->set( array(
                        'tag_group_server_side_render' => $tag_group_server_side_render,
                    ) );
                    $view->render();
                    break;
                    // filters
                // filters
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * renders a settings page: front end
         *
         * @param void
         * @return void
         */
        public static function settings_page_front_end()
        {
            global  $tag_groups_premium_fs_sdk ;
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            $default_themes = explode( ',', TAG_GROUPS_BUILT_IN_THEMES );
            $tag_group_theme = TagGroups_Options::get_option( 'tag_group_theme', TAG_GROUPS_STANDARD_THEME );
            $tag_group_mouseover = TagGroups_Options::get_option( 'tag_group_mouseover', 0 );
            $tag_group_collapsible = TagGroups_Options::get_option( 'tag_group_collapsible', 0 );
            $tag_group_enqueue_jquery = TagGroups_Options::get_option( 'tag_group_enqueue_jquery', 1 );
            $tag_group_html_description = TagGroups_Options::get_option( 'tag_group_html_description', 0 );
            $tag_group_shortcode_widget = TagGroups_Options::get_option( 'tag_group_shortcode_widget' );
            $tag_group_shortcode_enqueue_always = TagGroups_Options::get_option( 'tag_group_shortcode_enqueue_always', 1 );
            self::add_header();
            self::add_settings_help();
            $tabs = array();
            $tabs['shortcodes'] = __( 'Shortcodes', 'tag-groups' );
            $tabs['themes'] = __( 'Themes and Appearance', 'tag-groups' );
            $tabs = apply_filters( 'tag_groups_settings_front_end_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-front-end', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'shortcodes':
                    /**
                     * Let the premium plugin add own shortcode information.
                     */
                    $premium_shortcode_info = apply_filters( 'tag_groups_hook_shortcodes', '' );
                    $view = new TagGroups_View( 'admin/settings_front_end_shortcodes' );
                    $gutenberg_documentation_link = 'https://documentation.chattymango.com/documentation/tag-groups/tag-clouds-and-groups-info/using-gutenberg/?pk_campaign=tg&pk_kwd=settings';
                    $view->set( array(
                        'premium_shortcode_info'             => $premium_shortcode_info,
                        'tag_group_shortcode_enqueue_always' => $tag_group_shortcode_enqueue_always,
                        'tag_group_shortcode_widget'         => $tag_group_shortcode_widget,
                        'gutenberg_documentation_link'       => $gutenberg_documentation_link,
                    ) );
                    $view->render();
                    break;
                case 'themes':
                    $view = new TagGroups_View( 'admin/settings_front_end_themes' );
                    $view->set( array(
                        'default_themes'             => $default_themes,
                        'tag_group_theme'            => $tag_group_theme,
                        'tag_group_enqueue_jquery'   => $tag_group_enqueue_jquery,
                        'tag_group_mouseover'        => $tag_group_mouseover,
                        'tag_group_collapsible'      => $tag_group_collapsible,
                        'tag_group_html_description' => $tag_group_html_description,
                    ) );
                    $view->render();
                    break;
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * renders a settings page: tools
         *
         * @param void
         * @return void
         */
        public static function settings_page_tools()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            $tag_group_reset_when_uninstall = TagGroups_Options::get_option( 'tag_group_reset_when_uninstall', 0 );
            self::add_header();
            self::add_settings_help();
            $tabs = array();
            $tabs['export_import'] = __( 'Export/Import', 'tag-groups' );
            $tabs['reset'] = __( 'Reset', 'tag-groups' );
            $tabs = apply_filters( 'tag_groups_settings_tools_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-tools', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'export_import':
                    $view = new TagGroups_View( 'admin/settings_tools_export_import' );
                    $view->render();
                    break;
                case 'reset':
                    $view = new TagGroups_View( 'admin/settings_tools_reset' );
                    $view->set( 'tag_group_reset_when_uninstall', $tag_group_reset_when_uninstall );
                    $view->render();
                    break;
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * renders a settings page: troubleshooting
         *
         * @param void
         * @return void
         */
        public static function settings_page_troubleshooting()
        {
            global  $tag_group_groups, $tag_groups_premium_fs_sdk ;
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            self::add_header();
            self::add_settings_help();
            $tabs = array();
            $tabs['first-aid'] = __( 'First Aid', 'tag-groups' );
            $tabs['faq'] = __( 'FAQ and Common Issues', 'tag-groups' );
            $tabs['documentation'] = __( 'Documentation', 'tag-groups' );
            $tabs['support'] = __( 'Get Support', 'tag-groups' );
            $tabs['system'] = __( 'System Information', 'tag-groups' );
            $tabs['debug'] = __( 'Debugging', 'tag-groups' );
            $tabs = apply_filters( 'tag_groups_settings_troubleshooting_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-troubleshooting', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'first-aid':
                    
                    if ( !empty($_GET['process-tasks']) ) {
                        self::add_html_process();
                    } else {
                        $view = new TagGroups_View( 'admin/settings_troubleshooting_first_aid' );
                        $view->set( 'tasks_migration', 'migratetermmeta' );
                        $view->set( 'tasks_maintenance', 'fixgroups,fixmissinggroups,sortgroups' );
                        $view->set( 'tag_group_show_filter_tags', TagGroups_Options::get_option( 'tag_group_show_filter_tags', 1 ) );
                        $view->render();
                    }
                    
                    break;
                case 'faq':
                    $view = new TagGroups_View( 'admin/settings_troubleshooting_faq' );
                    $view->render();
                    break;
                case 'documentation':
                    $view = new TagGroups_View( 'admin/settings_troubleshooting_documentation' );
                    $view->render();
                    break;
                case 'support':
                    $view = new TagGroups_View( 'admin/settings_troubleshooting_support' );
                    $view->render();
                    break;
                case 'system':
                    $phpversion = phpversion();
                    
                    if ( version_compare( $phpversion, '7.0.0', '<' ) ) {
                        $php_upgrade_recommendation = true;
                    } else {
                        $php_upgrade_recommendation = false;
                    }
                    
                    $active_theme = wp_get_theme();
                    $protocol = ( isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://' );
                    $ajax_test_url = admin_url( 'admin-ajax.php', $protocol );
                    /* constants */
                    $wp_constants = array(
                        'WP_DEBUG',
                        'WP_DEBUG_DISPLAY',
                        'WP_DEBUG_LOG',
                        'ABSPATH',
                        // 'WP_HOME',
                        'MULTISITE',
                        'WP_CACHE',
                        'COMPRESS_SCRIPTS',
                        // 'FS_CHMOD_DIR',
                        // 'FS_CHMOD_FILE',
                        'FORCE_SSL_ADMIN',
                        'CM_UPDATE_CHECK',
                        'WP_MEMORY_LIMIT',
                        'WP_MAX_MEMORY_LIMIT',
                    );
                    sort( $wp_constants );
                    $constants = get_defined_constants();
                    foreach ( $constants as &$constant ) {
                        if ( isset( $constant ) ) {
                            $constant = self::echo_var( $constant );
                        }
                    }
                    ksort( $constants );
                    $benchmarks = array();
                    $benchmark['name'] = 'Database: tags (1000x read)';
                    $group_ids = $tag_group_groups->get_group_ids();
                    $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
                    $start_time = microtime( TRUE );
                    for ( $i = 0 ;  $i < 1000 ;  $i++ ) {
                        $group_id = $group_ids[array_rand( $group_ids )];
                        $tg_group = new TagGroups_Group( $group_id );
                        $group_terms_ids_dummy = $tg_group->get_group_terms( $enabled_taxonomies, false, 'ids' );
                    }
                    $benchmark['value'] = sprintf( '%d ms', 1000 * (microtime( TRUE ) - $start_time) );
                    $benchmarks[] = $benchmark;
                    /**
                     * Prepare the cache here so that we can test if it persists beyond seesions
                     */
                    $tag_group_object_cache = TagGroups_Options::get_option( 'tag_group_object_cache', TagGroups_Object_Cache::WP_TRANSIENTS );
                    $object_cache_options = array(
                        1 => __( 'Transients', 'tag-groups' ),
                        2 => __( 'Database', 'tag-groups' ),
                        3 => __( 'Filesystem', 'tag-groups' ),
                        9 => __( 'WP Object Cache', 'tag-groups' ),
                    );
                    $cache_key = md5( 'benchmark' );
                    foreach ( $object_cache_options as $object_cache_option_id => $object_cache_option_name ) {
                        update_option( 'tag_group_object_cache', $object_cache_option_id );
                        do_action( 'tag_groups_hook_cache_set', $cache_key . '-efficacy-test', '' );
                    }
                    update_option( 'tag_group_object_cache', $tag_group_object_cache );
                    if ( !defined( 'TAG_GROUPS_PLUGIN_IS_FREE' ) || !TAG_GROUPS_PLUGIN_IS_FREE ) {
                    }
                    global  $wpdb ;
                    $db_info = $wpdb->db_server_info();
                    if ( empty($db_info) ) {
                        $db_info = 'unknown';
                    }
                    $view = new TagGroups_View( 'admin/settings_troubleshooting_system' );
                    $view->set( array(
                        'phpversion'                 => $phpversion,
                        'php_upgrade_recommendation' => $php_upgrade_recommendation,
                        'db_info'                    => $db_info,
                        'wp_constants'               => $wp_constants,
                        'constants'                  => $constants,
                        'ajax_test_url'              => $ajax_test_url,
                        'active_theme'               => $active_theme,
                        'benchmarks'                 => $benchmarks,
                        'tag_groups_premium_fs_sdk'  => $tag_groups_premium_fs_sdk,
                    ) );
                    $view->render();
                    break;
                case 'debug':
                    $help_url = 'https://documentation.chattymango.com/documentation/tag-groups/troubleshooting/how-to-use-the-debug-log-2/?pk_campaign=tg&pk_kwd=settings';
                    $view = new TagGroups_View( 'admin/settings_troubleshooting_debug' );
                    $verbose_is_on_hardcoded = defined( 'CM_DEBUG' ) && strtolower( CM_DEBUG ) == 'verbose';
                    $verbose_is_on_option = (bool) TagGroups_Options::get_option( 'tag_group_verbose_debug', 0 );
                    $view->set( array(
                        'debug_is_on'             => defined( 'WP_DEBUG' ) && WP_DEBUG,
                        'verbose_is_on'           => defined( 'CM_DEBUG' ) && $verbose_is_on_hardcoded || $verbose_is_on_option,
                        'help_url'                => $help_url,
                        'verbose_is_on_hardcoded' => $verbose_is_on_hardcoded,
                    ) );
                    $view->render();
                    break;
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * Runs selected routines for benchmarking
         *
         *
         * @param void
         * @return string HTML
         * @since 1.23.0
         */
        public static function ajax_benchmark()
        {
            global  $tag_groups_premium_fs_sdk ;
            $benchmark = array();
            switch ( $_POST['task'] ) {
                case "cache":
                    break;
            }
            echo  json_encode( $benchmark ) ;
            exit;
        }
        
        /**
         * renders a settings page: premium
         *
         * @param void
         * @return void
         */
        public static function settings_page_premium()
        {
            global  $tag_groups_premium_fs_sdk ;
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            self::add_header();
            $view = new TagGroups_View( 'admin/settings_premium' );
            $view->set( 'tag_groups_premium_fs_sdk', $tag_groups_premium_fs_sdk );
            $view->render();
            self::add_footer();
        }
        
        /**
         * renders a settings page: about
         *
         * @param void
         * @return void
         */
        public static function settings_page_about()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            self::add_header();
            self::add_settings_help();
            $tabs = array();
            $tabs['info'] = __( 'Info', 'tag-groups' );
            $tabs['licenses'] = __( 'Licenses', 'tag-groups' );
            $tabs['news'] = __( 'Development News', 'tag-groups' );
            $tabs = apply_filters( 'tag_groups_settings_about_tabs', $tabs );
            $active_tab = self::get_active_tab( $tabs );
            self::add_tabs( 'tag-groups-settings-about', $tabs, $active_tab );
            switch ( $active_tab ) {
                case 'info':
                    $view = new TagGroups_View( 'admin/settings_about_info' );
                    $view->render();
                    break;
                case 'licenses':
                    $view = new TagGroups_View( 'admin/settings_about_licenses' );
                    $view->render();
                    break;
                case 'news':
                    $protocol = ( isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://' );
                    $admin_url = admin_url( 'admin-ajax.php', $protocol );
                    $view = new TagGroups_View( 'admin/settings_about_news' );
                    $view->set( 'admin_url', $admin_url );
                    $view->render();
                    break;
                default:
                    if ( class_exists( 'TagGroups_Premium_Settings' ) ) {
                        TagGroups_Premium_Settings::get_content( $active_tab );
                    }
                    break;
            }
            self::add_footer();
        }
        
        /**
         * renders a menu-less settings page: onboarding
         *
         * @param void
         * @return void
         */
        public static function settings_page_onboarding()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                return;
            }
            global  $tag_groups_premium_fs_sdk ;
            self::add_header();
            $settings_taxonomy_link = admin_url( 'admin.php?page=tag-groups-settings-taxonomies' );
            $settings_home_link = admin_url( 'admin.php?page=tag-groups-settings' );
            $settings_premium_link = admin_url( 'admin.php?page=tag-groups-settings-premium' );
            $settings_setup_wizard_link = admin_url( 'admin.php?page=tag-groups-settings-setup-wizard' );
            
            if ( defined( 'TAG_GROUPS_PLUGIN_IS_FREE' ) && TAG_GROUPS_PLUGIN_IS_FREE ) {
                $title = 'Tag Groups';
                $documentation_link = 'https://documentation.chattymango.com/documentation/tag-groups/';
                $logo = '<img src="' . TAG_GROUPS_PLUGIN_URL . '/assets/images/cm-tg-icon-64x64.png" alt="Tag Groups logo" class="tg_onboarding_logo"/>';
            } else {
                $title = 'Tag Groups Premium';
                $documentation_link = 'https://documentation.chattymango.com/documentation/tag-groups-premium/';
                $logo = '<img src="' . TAG_GROUPS_PLUGIN_URL . '/assets/images/cm-tgp-icon-64x64.png" alt="Tag Groups Premium logo" class="tg_onboarding_logo"/>';
            }
            
            $view = new TagGroups_View( 'admin/onboarding' );
            $view->set( array(
                'logo'                       => $logo,
                'title'                      => $title,
                'settings_taxonomy_link'     => $settings_taxonomy_link,
                'settings_home_link'         => $settings_home_link,
                'documentation_link'         => $documentation_link,
                'settings_premium_link'      => $settings_premium_link,
                'settings_setup_wizard_link' => $settings_setup_wizard_link,
            ) );
            $view->render();
            self::add_footer();
        }
        
        /**
         * renders a menu-less settings page: onboarding
         *
         * @param void
         * @return void
         */
        public static function settings_page_setup_wizard()
        {
            // Make very sure that only administrators can access this page
            if ( !current_user_can( 'manage_options' ) ) {
                return;
            }
            global  $tag_groups_premium_fs_sdk, $tag_group_groups ;
            self::add_header();
            $step = ( isset( $_GET['step'] ) && $_GET['step'] > 0 ? (int) $_GET['step'] : 1 );
            $setup_wizard_next_link = add_query_arg( 'step', $step + 1, admin_url( 'admin.php?page=tag-groups-settings-setup-wizard' ) );
            
            if ( defined( 'TAG_GROUPS_PLUGIN_IS_FREE' ) && TAG_GROUPS_PLUGIN_IS_FREE ) {
                $title = 'Tag Groups';
                $is_premium = false;
                $documentation_link = 'https://documentation.chattymango.com/documentation/tag-groups/';
            } else {
                $title = 'Tag Groups Premium';
                $is_premium = true;
                $documentation_link = 'https://documentation.chattymango.com/documentation/tag-groups-premium/';
            }
            
            
            if ( $is_premium && $tag_groups_premium_fs_sdk->is_plan_or_trial( 'premium' ) && class_exists( 'TagGroups_Premium_View' ) ) {
            } else {
                $steps = array(
                    1 => array(
                    'id'    => 'start',
                    'title' => 'Start',
                ),
                    2 => array(
                    'id'    => 'taxonomies',
                    'title' => 'Taxonomies',
                ),
                    3 => array(
                    'id'    => 'sample_content',
                    'title' => 'Sample Content',
                ),
                    4 => array(
                    'id'    => 'finished',
                    'title' => null,
                ),
                );
            }
            
            $view = new TagGroups_View( 'admin/setup_wizard_header' );
            $view->set( array(
                'title' => $title,
                'step'  => $step,
                'steps' => $steps,
            ) );
            $view->render();
            switch ( $steps[$step]['id'] ) {
                case 'sample_content':
                    $view = new TagGroups_View( 'admin/setup_wizard_sample_content' );
                    $group_names = array( 'Sample Group A', 'Sample Group B', 'Sample Group C' );
                    /**
                     * Make sure they don't yet exist
                     */
                    $group_names = array_map( function ( $original_name ) {
                        $tg_group = new TagGroups_Group();
                        $name = $original_name;
                        $i = 0;
                        while ( $tg_group->find_by_label( $name ) !== false ) {
                            $i++;
                            $name = $original_name . ' - ' . $i;
                        }
                        return $name;
                    }, $group_names );
                    $tag_names = array( 'First Sample Tag', 'Second Sample Tag', 'Third Sample Tag' );
                    $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
                    $taxonomy = array_shift( $enabled_taxonomies );
                    /**
                     * Make sure they don't yet exist
                     */
                    $tag_names = array_map( function ( $original_name ) use( $taxonomy ) {
                        $name = $original_name;
                        $i = 0;
                        while ( get_term_by( 'name', $name, $taxonomy ) !== false ) {
                            $i++;
                            $name = $original_name . ' - ' . $i;
                        }
                        return $name;
                    }, $tag_names );
                    
                    if ( TagGroups_Gutenberg::is_gutenberg_active() ) {
                        $create_sample_page_label = __( 'Create a draft sample page with Gutenberg blocks.', 'tag-groups' );
                    } else {
                        $create_sample_page_label = __( 'Create a draft sample page with shortcodes.', 'tag-groups' );
                    }
                    
                    $view->set( array(
                        'title'                    => $title,
                        'group_names'              => $group_names,
                        'tag_names'                => $tag_names,
                        'create_sample_page_label' => $create_sample_page_label,
                        'setup_wizard_next_link'   => $setup_wizard_next_link,
                    ) );
                    break;
                case 'post_tags':
                    break;
                case 'meta_box':
                    break;
                case 'taxonomies':
                    $view = new TagGroups_View( 'admin/setup_wizard_taxonomies' );
                    $view->set( array(
                        'title'                  => $title,
                        'public_taxonomies'      => TagGroups_Taxonomy::get_public_taxonomies(),
                        'enabled_taxonomies'     => TagGroups_Taxonomy::get_enabled_taxonomies(),
                        'setup_wizard_next_link' => $setup_wizard_next_link,
                    ) );
                    break;
                case 'finished':
                    $view = new TagGroups_View( 'admin/setup_wizard_finished' );
                    $documentation_link = ( $is_premium ? 'https://documentation.chattymango.com/documentation/tag-groups-premium/?pk_campaign=tgp&pk_kwd=wizard' : 'https://documentation.chattymango.com/documentation/tag-groups/?pk_campaign=tg&pk_kwd=wizard' );
                    $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
                    $taxonomy = array_shift( $enabled_taxonomies );
                    $view->set( array(
                        'groups_admin_link'        => TagGroups_Taxonomy::get_tag_group_admin_url( $taxonomy ),
                        'documentation_link'       => $documentation_link,
                        'settings_home_link'       => admin_url( 'admin.php?page=tag-groups-settings' ),
                        'tag_group_sample_page_id' => TagGroups_Options::get_option( 'tag_group_sample_page_id', 0 ),
                    ) );
                    break;
                case 'start':
                default:
                    $view = new TagGroups_View( 'admin/setup_wizard_start' );
                    $view->set( array(
                        'title'                  => $title,
                        'setup_wizard_next_link' => $setup_wizard_next_link,
                        'is_premium'             => $is_premium,
                    ) );
                    break;
            }
            $view->render();
            $view = new TagGroups_View( 'admin/setup_wizard_footer' );
            $view->render();
            self::add_footer();
        }
        
        /**
         * Processes form submissions from the settings page
         *
         * @param void
         * @return void
         */
        static function settings_page_actions_wizard()
        {
            global  $tag_group_groups, $tag_groups_premium_fs_sdk ;
            if ( empty($_REQUEST['tg_action_wizard']) ) {
                return;
            }
            // Make very sure that only administrators can do actions
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            if ( !isset( $_POST['tag-groups-setup-wizard-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-setup-wizard-nonce'], 'tag-groups-setup-wizard-nonce' ) ) {
                die( "Security check failed" );
            }
            $enabled_taxonomies = TagGroups_Taxonomy::get_enabled_taxonomies();
            $taxonomy = array_shift( $enabled_taxonomies );
            switch ( $_REQUEST['tg_action_wizard'] ) {
                case 'taxonomy':
                    
                    if ( isset( $_POST['taxonomies'] ) ) {
                        $taxonomies = $_POST['taxonomies'];
                        
                        if ( is_array( $taxonomies ) ) {
                            $taxonomies = array_map( 'sanitize_text_field', $taxonomies );
                            $taxonomies = array_map( 'stripslashes', $taxonomies );
                        } else {
                            $taxonomies = array( 'post_tag' );
                        }
                    
                    } else {
                        $taxonomies = array( 'post_tag' );
                    }
                    
                    $public_taxonomies = TagGroups_Taxonomy::get_public_taxonomies();
                    foreach ( $taxonomies as $taxonomy_item ) {
                        if ( !in_array( $taxonomy_item, $public_taxonomies ) ) {
                            return;
                        }
                    }
                    TagGroups_Taxonomy::update_enabled( $taxonomies );
                    break;
                case 'sample-content':
                    $created_groups = array();
                    /**
                     * Create groups
                     */
                    if ( isset( $_POST['tag-groups-create-sample-groups'] ) && $_POST['tag-groups-create-sample-groups'] ) {
                        foreach ( $_POST['tag_groups_group_names'] as $group_name ) {
                            $tg_group = new TagGroups_Group();
                            $tg_group->create( sanitize_text_field( $group_name ) );
                            $created_groups[] = $tg_group->get_group_id();
                        }
                    }
                    /**
                     * Create tags
                     */
                    if ( isset( $_POST['tag-groups-create-sample-tags'] ) && $_POST['tag-groups-create-sample-tags'] ) {
                        foreach ( $_POST['tag_groups_tag_names'] as $tag_name ) {
                            $tag_name = sanitize_text_field( $tag_name );
                            
                            if ( !term_exists( $tag_name, $taxonomy ) ) {
                                $term_array = wp_insert_term( $tag_name, $taxonomy );
                                $tg_term = new TagGroups_Term( $term_array['term_id'] );
                                
                                if ( empty($created_groups) ) {
                                    $group_ids = $tag_group_groups->get_group_ids();
                                    unset( $group_ids[0] );
                                } else {
                                    $group_ids = $created_groups;
                                }
                                
                                // add one group
                                $amount = 1;
                                
                                if ( 1 == $amount ) {
                                    $random_group_ids = $group_ids[array_rand( $group_ids )];
                                } else {
                                    $random_group_ids = array_intersect_key( $group_ids, array_rand( $group_ids, $amount ) );
                                }
                                
                                $tg_term->add_group( $random_group_ids )->save();
                            }
                        
                        }
                    }
                    $tpf_include = $tag_group_groups->get_group_ids();
                    unset( $tpf_include[0] );
                    
                    if ( isset( $_POST['tag-groups-create-sample-page'] ) && $_POST['tag-groups-create-sample-page'] ) {
                        
                        if ( TagGroups_Gutenberg::is_gutenberg_active() ) {
                            $view = new TagGroups_View( 'admin/sample_page_gutenberg' );
                            $sample_page_title = 'Tag Groups (Free) Sample Page - Gutenberg Editor';
                        } else {
                            $view = new TagGroups_View( 'admin/sample_page' );
                            $sample_page_title = 'Tag Groups (Free) Sample Page - Classic Editor';
                        }
                        
                        $tag_groups_settings_link = admin_url( 'admin.php?page=tag-groups-settings' );
                        $current_user = wp_get_current_user();
                        $view->set( array(
                            'enabled_taxonomies'        => $enabled_taxonomies,
                            'author_display_name'       => $current_user->display_name,
                            'tag_groups_settings_link'  => $tag_groups_settings_link,
                            'tag_groups_premium_fs_sdk' => $tag_groups_premium_fs_sdk,
                            'tpf_include_csv'           => implode( ',', $tpf_include ),
                        ) );
                        $content = $view->return_html();
                        $post_data = array(
                            'post_title'   => wp_strip_all_tags( $sample_page_title ),
                            'post_content' => $content,
                            'post_status'  => 'draft',
                            'post_type'    => 'page',
                            'post_author'  => get_current_user_id(),
                        );
                        $post_id = wp_insert_post( $post_data );
                        update_option( 'tag_group_sample_page_id', $post_id );
                    } else {
                        delete_option( 'tag_group_sample_page_id' );
                    }
                    
                    break;
            }
        }
        
        /**
         * Processes form submissions from the settings page
         *
         * @param void
         * @return void
         */
        static function settings_page_actions()
        {
            global  $tag_group_groups ;
            
            if ( !empty($_REQUEST['tg_action']) ) {
                $tg_action = $_REQUEST['tg_action'];
            } else {
                return;
            }
            
            // Make very sure that only administrators can do actions
            if ( !current_user_can( 'manage_options' ) ) {
                wp_die( "Capability check failed" );
            }
            
            if ( isset( $_POST['ok'] ) ) {
                $ok = $_POST['ok'];
            } else {
                $ok = '';
            }
            
            switch ( $tg_action ) {
                case 'shortcode':
                    if ( !isset( $_POST['tag-groups-shortcode-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-shortcode-nonce'], 'tag-groups-shortcode' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    
                    if ( isset( $_POST['widget'] ) && $_POST['widget'] == '1' ) {
                        update_option( 'tag_group_shortcode_widget', 1 );
                    } else {
                        update_option( 'tag_group_shortcode_widget', 0 );
                    }
                    
                    
                    if ( isset( $_POST['enqueue'] ) && $_POST['enqueue'] == '1' ) {
                        update_option( 'tag_group_shortcode_enqueue_always', 1 );
                    } else {
                        update_option( 'tag_group_shortcode_enqueue_always', 0 );
                    }
                    
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                case 'reset':
                    if ( !isset( $_POST['tag-groups-reset-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-reset-nonce'], 'tag-groups-reset' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    
                    if ( $ok == 'yes' ) {
                        $tag_group_groups->reset_groups();
                        /**
                         * Remove filters
                         */
                        delete_option( 'tag_group_tags_filter' );
                        TagGroups_Admin_Notice::add( 'success', __( 'All groups have been deleted and assignments reset.', 'tag-groups' ) );
                    }
                    
                    break;
                case 'uninstall':
                    if ( !isset( $_POST['tag-groups-uninstall-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-uninstall-nonce'], 'tag-groups-uninstall' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    
                    if ( $ok == 'yes' ) {
                        update_option( 'tag_group_reset_when_uninstall', 1 );
                    } else {
                        update_option( 'tag_group_reset_when_uninstall', 0 );
                    }
                    
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                case 'theme':
                    if ( !isset( $_POST['tag-groups-settings-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-settings-nonce'], 'tag-groups-settings' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    $theme = '';
                    if ( isset( $_POST['theme'] ) ) {
                        switch ( $_POST['theme'] ) {
                            case 'own':
                                if ( isset( $_POST['theme-name'] ) ) {
                                    $theme = stripslashes( sanitize_text_field( $_POST['theme-name'] ) );
                                }
                                break;
                            case 'none':
                                $theme = '';
                                break;
                            default:
                                $theme = stripslashes( sanitize_text_field( $_POST['theme'] ) );
                                break;
                        }
                    }
                    update_option( 'tag_group_theme', $theme );
                    $mouseover = ( isset( $_POST['mouseover'] ) && $_POST['mouseover'] == '1' ? 1 : 0 );
                    $collapsible = ( isset( $_POST['collapsible'] ) && $_POST['collapsible'] == '1' ? 1 : 0 );
                    $html_description = ( isset( $_POST['html_description'] ) && $_POST['html_description'] == '1' ? 1 : 0 );
                    update_option( 'tag_group_mouseover', $mouseover );
                    update_option( 'tag_group_collapsible', $collapsible );
                    update_option( 'tag_group_html_description', $html_description );
                    $tag_group_enqueue_jquery = ( isset( $_POST['enqueue-jquery'] ) && $_POST['enqueue-jquery'] == '1' ? 1 : 0 );
                    update_option( 'tag_group_enqueue_jquery', $tag_group_enqueue_jquery );
                    // TagGroups_Admin::clear_cache();
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    do_action( 'tag_groups_theme_saved' );
                    break;
                case 'taxonomy':
                    if ( !isset( $_POST['tag-groups-taxonomy-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-taxonomy-nonce'], 'tag-groups-taxonomy' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    
                    if ( isset( $_POST['taxonomies'] ) ) {
                        $taxonomies = $_POST['taxonomies'];
                        
                        if ( is_array( $taxonomies ) ) {
                            $taxonomies = array_map( 'sanitize_text_field', $taxonomies );
                            $taxonomies = array_map( 'stripslashes', $taxonomies );
                        } else {
                            $taxonomies = array( 'post_tag' );
                        }
                    
                    } else {
                        $taxonomies = array( 'post_tag' );
                    }
                    
                    $public_taxonomies = TagGroups_Taxonomy::get_public_taxonomies();
                    foreach ( $taxonomies as $taxonomy_item ) {
                        if ( !in_array( $taxonomy_item, $public_taxonomies ) ) {
                            die( "Security check: taxonomies" );
                        }
                    }
                    TagGroups_Taxonomy::update_enabled( $taxonomies );
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                case 'backend':
                    if ( !isset( $_POST['tag-groups-backend-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-backend-nonce'], 'tag-groups-backend' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    $show_filter_posts = ( isset( $_POST['filter_posts'] ) ? 1 : 0 );
                    update_option( 'tag_group_show_filter', $show_filter_posts );
                    $show_filter_tags = ( isset( $_POST['filter_tags'] ) ? 1 : 0 );
                    update_option( 'tag_group_show_filter_tags', $show_filter_tags );
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                case 'gutenberg':
                    if ( !isset( $_POST['tag-groups-backend-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-backend-nonce'], 'tag-groups-backend' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    $tag_group_server_side_render = ( isset( $_POST['tag_group_server_side_render'] ) ? 1 : 0 );
                    update_option( 'tag_group_server_side_render', $tag_group_server_side_render );
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                case 'reset-tag-filter':
                    // check nonce
                    if ( !isset( $_POST['tag-groups-reset-tag-filter-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-reset-tag-filter-nonce'], 'tag-groups-reset-tag-filter' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    /**
                     * Reset the group filter above the tags list
                     */
                    update_option( 'tag_group_tags_filter', array() );
                    TagGroups_Admin_Notice::add( 'success', __( 'The filter on the tags page has been reset to show all tags.', 'tag-groups' ) );
                    break;
                case 'export':
                    if ( !isset( $_POST['tag-groups-export-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-export-nonce'], 'tag-groups-export' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    $export = new TagGroups_Export();
                    $export->process_options_for_export();
                    $export->process_terms_for_export();
                    $export->write_files();
                    $export->show_download_links();
                    break;
                case 'import':
                    if ( !isset( $_POST['tag-groups-import-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-import-nonce'], 'tag-groups-import' ) ) {
                        die( "Security check" );
                    }
                    // Make very sure that only administrators can upload stuff
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    if ( !function_exists( 'wp_handle_upload' ) ) {
                        require_once ABSPATH . 'wp-admin/includes/file.php';
                    }
                    $import = new TagGroups_Import();
                    $import->determine_file_type();
                    $import->read_file();
                    $import->parse_and_save();
                    break;
                case 'debug':
                    if ( !isset( $_POST['tag-groups-debug-nonce'] ) || !wp_verify_nonce( $_POST['tag-groups-debug-nonce'], 'tag-groups-debug' ) ) {
                        die( "Security check" );
                    }
                    // Make sure that only administrators can save settings
                    if ( !current_user_can( 'manage_options' ) ) {
                        wp_die( "Capability check failed" );
                    }
                    $verbose_debug = ( '1' == $_POST['verbose_debug'] ? 1 : 0 );
                    update_option( 'tag_group_verbose_debug', $verbose_debug );
                    
                    if ( $verbose_debug ) {
                        TagGroups_Error::log( '[Tag Groups] Verbose logging has been turned on.' );
                    } else {
                        TagGroups_Error::log( '[Tag Groups] Verbose logging has been turned off.' );
                    }
                    
                    TagGroups_Admin_Notice::add( 'success', __( 'Your settings have been saved.', 'tag-groups' ) );
                    break;
                default:
                    // hook for premium plugin
                    do_action( 'tag_groups_hook_settings_action', $tg_action );
                    break;
            }
        }
        
        /**
         * Prepares variable for echoing as string
         *
         *
         * @param mixed $var Mixed type that needs to be echoed as string.
         * @return return string
         */
        private static function echo_var( $var = null )
        {
            
            if ( is_bool( $var ) ) {
                return ( $var ? 'true' : 'false' );
            } elseif ( is_array( $var ) ) {
                return print_r( $var, true );
            } else {
                return (string) $var;
            }
        
        }
        
        /**
         * Returns an array that contains topics covered in the settings
         *
         * @param void
         * @return array
         */
        public static function get_setting_topics()
        {
            $public_taxonomies_slugs = TagGroups_Taxonomy::get_public_taxonomies();
            $public_taxonomies_names = array_map( array( 'TagGroups_Taxonomy', 'get_name_from_slug' ), $public_taxonomies_slugs );
            $topics = array(
                'taxonomies'      => array(
                'title'    => __( 'Taxonomies', 'tag-groups' ),
                'page'     => 'tag-groups-settings-taxonomies',
                'keywords' => array_merge( array_keys( $public_taxonomies_names ), array_values( $public_taxonomies_names ), array( __( 'tag groups', 'tag-groups' ) ) ),
            ),
                'shortcodes'      => array(
                'title'    => __( 'Shortcodes', 'tag-groups' ),
                'page'     => 'tag-groups-settings-front-end',
                'keywords' => array(
                __( 'tag cloud', 'tag-groups' ),
                __( 'group info', 'tag-groups' ),
                __( 'sidebar widget', 'tag-groups' ),
                __( 'accordion', 'tag-groups' ),
                __( 'tabs', 'tag-groups' ),
                __( 'alphabetical', 'tag-groups' ),
                __( 'post list', 'tag-groups' ),
                'Gutenberg'
            ),
            ),
                'themes'          => array(
                'title'    => __( 'Themes and Appearance', 'tag-groups' ),
                'page'     => 'tag-groups-settings-front-end',
                'keywords' => array(
                __( 'tag cloud', 'tag-groups' ),
                'CSS',
                'style',
                'HTML',
                __( 'colors', 'tag-groups' ),
                __( 'tag description', 'tag-groups' )
            ),
            ),
                'filters'         => array(
                'title'    => __( 'Filters', 'tag-groups' ),
                'page'     => 'tag-groups-settings-back-end',
                'keywords' => array( __( 'tag filter', 'tag-groups' ), __( 'post filter', 'tag-groups' ) ),
            ),
                'export_import'   => array(
                'title'    => __( 'Export/Import', 'tag-groups' ),
                'page'     => 'tag-groups-settings-tools',
                'keywords' => array( __( 'backup', 'tag-groups' ) ),
            ),
                'reset'           => array(
                'title'    => __( 'Reset', 'tag-groups' ),
                'page'     => 'tag-groups-settings-tools',
                'keywords' => array( __( 'remove plugin', 'tag-groups' ), __( 'remove data', 'tag-groups' ), __( 'delete groups', 'tag-groups' ) ),
            ),
                'faq'             => array(
                'title'    => __( 'FAQ and Common Issues', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array(
                __( 'frequently asked questions', 'tag-groups' ),
                __( 'help', 'tag-groups' ),
                __( 'bug', 'tag-groups' ),
                __( 'problem', 'tag-groups' ),
                __( 'troubleshooting', 'tag-groups' ),
                __( 'support', 'tag-groups' )
            ),
            ),
                'documentation'   => array(
                'title'    => __( 'Documentation', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array(
                __( 'instructions', 'tag-groups' ),
                __( 'help', 'tag-groups' ),
                __( 'problem', 'tag-groups' ),
                __( 'troubleshooting', 'tag-groups' ),
                __( 'support', 'tag-groups' ),
                'Gutenberg',
                'CSS',
                'style',
                'PHP',
                'REST API'
            ),
            ),
                'support'         => array(
                'title'    => __( 'Get Support', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array(
                __( 'support', 'tag-groups' ),
                __( 'contact', 'tag-groups' ),
                __( 'forum', 'tag-groups' ),
                __( 'bug', 'tag-groups' ),
                __( 'problem', 'tag-groups' ),
                __( 'help', 'tag-groups' )
            ),
            ),
                'system'          => array(
                'title'    => __( 'System Information', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array(
                __( 'debugging', 'tag-groups' ),
                __( 'PHP Version', 'tag-groups' ),
                __( 'Ajax Test', 'tag-groups' ),
                __( 'troubleshooting', 'tag-groups' ),
                __( 'benchmarks', 'tag-groups' ),
                __( 'speed test', 'tag-groups' ),
                __( 'error', 'tag-groups' )
            ),
            ),
                'debug'           => array(
                'title'    => __( 'Debugging', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array( __( 'debugging', 'tag-groups' ), __( 'troubleshooting', 'tag-groups' ), __( 'error', 'tag-groups' ) ),
            ),
                'premium'         => array(
                'title'    => __( 'Premium', 'tag-groups' ),
                'page'     => 'tag-groups-settings-premium',
                'keywords' => array(
                __( 'upgrade', 'tag-groups' ),
                __( 'more groups', 'tag-groups' ),
                __( 'posts', 'tag-groups' ),
                __( 'tag cloud', 'tag-groups' ),
                __( 'filter', 'tag-groups' ),
                __( 'animated', 'tag-groups' ),
                __( 'searchable', 'tag-groups' ),
                'Shuffle Box',
                'Toggle Post Filter',
                'Dynamic Post Filter',
                'WooCommerce'
            ),
            ),
                'info'            => array(
                'title'    => __( 'Info', 'tag-groups' ),
                'page'     => 'tag-groups-settings-about',
                'keywords' => array(
                __( 'author', 'tag-groups' ),
                __( 'version', 'tag-groups' ),
                __( 'contact', 'tag-groups' ),
                __( 'about', 'tag-groups' )
            ),
            ),
                'licenses'        => array(
                'title'    => __( 'Licenses', 'tag-groups' ),
                'page'     => 'tag-groups-settings-about',
                'keywords' => array( __( 'Credits', 'tag-groups' ) ),
            ),
                'news'            => array(
                'title'    => __( 'Development News', 'tag-groups' ),
                'page'     => 'tag-groups-settings-about',
                'keywords' => array( __( 'blog', 'tag-groups' ), __( 'updates', 'tag-groups' ) ),
            ),
                'getting_started' => array(
                'title'    => __( 'First Steps', 'tag-groups' ),
                'page'     => 'tag-groups-settings-first-steps',
                'keywords' => array( __( 'getting started', 'tag-groups' ), __( 'introduction', 'tag-groups' ), __( 'help', 'tag-groups' ) ),
            ),
                'setup_wizard'    => array(
                'title'    => __( 'Setup Wizard', 'tag-groups' ),
                'page'     => 'tag-groups-settings-setup-wizard',
                'keywords' => array( __( 'getting started', 'tag-groups' ), __( 'introduction', 'tag-groups' ), __( 'sample', 'tag-groups' ) ),
            ),
                'first-aid'       => array(
                'title'    => __( 'First Aid', 'tag-groups' ),
                'page'     => 'tag-groups-settings-troubleshooting',
                'keywords' => array(
                __( 'troubleshooting', 'tag-groups' ),
                __( 'migrate', 'tag-groups' ),
                __( 'help', 'tag-groups' ),
                __( 'problem', 'tag-groups' ),
                __( 'tag filter', 'tag-groups' )
            ),
            ),
            );
            if ( TagGroups_Gutenberg::is_gutenberg_active() ) {
                $topics['gutenberg'] = array(
                    'title'    => __( 'Gutenberg', 'tag-groups' ),
                    'page'     => 'tag-groups-settings-back-end',
                    'keywords' => array( __( 'live block preview', 'tag-groups' ) ),
                );
            }
            $topics = apply_filters( 'tag_groups_setting_topics', $topics );
            return $topics;
        }
        
        /**
         * Renders the widget where you can search for help
         *
         * @param void
         * @return void
         */
        public static function add_settings_help()
        {
            $topics = self::get_setting_topics();
            asort( $topics );
            $view = new TagGroups_View( 'admin/settings_help' );
            $view->set( 'topics', $topics );
            $view->render();
        }
        
        /**
         * adds the processing of tasks
         *
         * @param void
         * @return string
         */
        public static function add_html_process()
        {
            $tasks_whitelist = array(
                'migratetermmeta'  => __( 'Migrating the term meta', 'tag-groups' ),
                'fixgroups'        => __( 'Fixing incorrect tag groups', 'tag-groups' ),
                'fixmissinggroups' => __( 'Fixing incorrect groups in term meta', 'tag-groups' ),
                'sortgroups'       => __( 'Sorting groups in term meta', 'tag-groups' ),
                'checkbadterms'    => __( 'Checking the tag names', 'tag-groups' ),
            );
            $totals = array();
            $languages = array();
            $tasks = explode( ',', $_GET['process-tasks'] );
            $tasks = array_map( 'sanitize_title', $tasks );
            $tasks = array_intersect( $tasks, array_keys( $tasks_whitelist ) );
            
            if ( !empty($_GET['task-set-name']) ) {
                $task_set_name = sanitize_text_field( $_GET['task-set-name'] );
            } else {
                $task_set_name = '';
            }
            
            $task_html = '';
            foreach ( $tasks as $key => $task ) {
                $totals[$task] = TagGroups_Process::get_task_total( $task );
                $view = new TagGroups_View( 'admin/settings_troubleshooting_process_task_no_language' );
                $view->set( array(
                    'task'            => $task,
                    'tasks_whitelist' => $tasks_whitelist,
                ) );
                $task_html .= $view->return_html();
            }
            /**
             * Add the Javascript part that loops through the tasks and through the chunks within each task
             *
             * Keep some timeout between the calls so that we don't block the browser
             * Parameters: chunk length and timeouts for chunk and task
             */
            $protocol = ( isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://' );
            $ajax_link = admin_url( 'admin-ajax.php', $protocol );
            
            if ( defined( 'TAG_GROUPS_CHUNK_SIZE' ) ) {
                $chunk_length = (int) TAG_GROUPS_CHUNK_SIZE;
            } else {
                $chunk_length = 30;
            }
            
            
            if ( defined( 'TAG_GROUPS_CHUNK_TIMEOUT' ) ) {
                $timeout_chunk = (int) TAG_GROUPS_CHUNK_TIMEOUT;
            } else {
                $timeout_chunk = 10 * 1000;
                // 10 seconds - for really slow networks
            }
            
            
            if ( defined( 'TAG_GROUPS_TASK_TIMEOUT' ) ) {
                $timeout_task = (int) TAG_GROUPS_TASK_TIMEOUT;
            } else {
                $timeout_task = 5 * 60 * 1000;
                // 5 minutes - can be long, but user needs to keep window open
            }
            
            /**
             * The result messages will be revealed by the Javascript routine
             */
            $view = new TagGroups_View( 'admin/settings_troubleshooting_process' );
            $view->set( array(
                'task_html'     => $task_html,
                'ajax_link'     => $ajax_link,
                'tasks'         => $tasks,
                'totals'        => $totals,
                'languages'     => $languages,
                'timeout_task'  => $timeout_task,
                'timeout_chunk' => $timeout_chunk,
                'chunk_length'  => $chunk_length,
                'task_set_name' => $task_set_name,
            ) );
            $view->render();
        }
    
    }
}